--[[FMPM package metadata:
info = "IP library"
location = "/usr/lib/ip.lua"
]]

local serialization = require "serialization"
local component = require "component"

local ip = {}
local SocketIP = {}
local ServerDTP = {}
local SocketDTP = {}

----------------
--  ServerDTP  --
----------------

ServerDTP._new = function(port)
    
end

ServerDTP.send = function(self, destHost, destPort, ...)
    
end

ServerDTP.recv = function(self)
    
end

ServerDTP.close = function(self)
    
end

-----------------
--  SocketDTP  --
-----------------

SocketDTP._new = function(inner, remote)
    local soc = {
        remote = remote,
        closed = false,
        _inner = inner,
        _queue = {},
    }
    setmetatable(soc, {
        __index = SocketDTP,
        __gc = SocketDTP.close})
    return soc
end

SocketDTP.rawSend = function(self, ...)
    self._inner:send(self.remote, ...)
end

SocketDTP.rawRecv = function(self)
    self._inner:recv()
end

SocketDTP.send = function(self, ...)
    
end

SocketDTP.recv = function(self)
    
end

SocketDTP.close = function(self)
    self:rawSend(serialization.serialize{type = "DTP", id="close"})
    self._inner:close()
end

----------------
--  SocketIP  --
----------------

SocketIP._new = function(port)
    port = math.floor(port)
    if port <= 0 or port >= 65535 then
        return nil, "Invalid port number."
    end
    
    local config = ip.getConfig()
    
    local soc = {
        port = port,
        hostname = config.hostname,
        gateway = config.gateway,
        closed = false,
        _queue = {},
    }
    setmetatable(soc, {
        __index = SocketIP,
        __gc = SocketIP.close})
    return soc
end

SocketIP.send = function(self, destHost, destPort, ...)
    checkArg(1, destPort, "number")
    checkArg(2, destHost, "string")
    destPort = math.floor(destPort)
    if destPort <= 0 or destPort > 65535 then
        return nil, "Invalid port number."
    end
    component.modem.send(self.gateway, 65535,
        serialization.serialize{type="IP", srcHost = self.hostname, destHost = destHost, srcPort = self.port, destPort = destPort, ttl=32},
        ...)
    return true
end

SocketIP.recv = function(self)
    return table.remove(self._queue, 1)
end

SocketIP.close = function(self)
    if not self.closed then
        self.closed = true
        ip._sockets[self.port] = nil
        component.modem.close(self.port)
    else
        return false
    end
end

------------
--  CORE  --
------------

ip._sockets = {}
setmetatable(ip._sockets, {__mode = "v"})

ip.getConfig = function()
    local file = io.open("/etc/ip.conf")
    if file == nil then
        error("Missing ip config file. Use `ipconfig` command to configure interface.")
    end
    config = serialization.unserialize(file:read("*a"))
    file:close()
    return config
end

ip.socket = function(port)
    checkArg(1, port, "number", "nil")
    if ip._sockets[port] ~= nil then
        return nil, "Port in use."
    end
    if port == nil or port == -1 then
        repeat
            port = math.random(49152, 65534)
        until ip._sockets[port] == nil
    end
    local soc, err = SocketIP._new(port)
    if soc then
        ip._sockets[port] = soc
        component.modem.open(port)
    end
    return soc, err
end

ip.server = function(port)
    checkArg(1, port, "number", "nil")
    local soc, err = ip.socket()
    if soc == nil then
        return nil, err
    end
    return ServerDTP._new(soc)
end

ip.connect = function(host, port, timeout)
    checkArg(1, host, "string")
    checkArg(2, port, "number")
    checkArg(3, timeout, "number", "nil")
    if timeout == nil then
        timeout = 1
    end
    local soc, err = ip.socket()
    if soc == nil then
        return nil, err
    end
    local soc, err = SocketDTP._new(soc, host)
    if soc == nil then
        return nil, err
    end
    soc:rawSend(serialization.serialize{type = "DTP", id="open"})
    local msg
    
    local time = computer.uptime() + timeout
    while computer.uptime() < time do
        msg = soc:rawRecv()
        if msg.type == "ICMP" then
            return nil, msg.id
        end
        if msg.type == "DTP" and msg.id="reject" then
            return nil, "Connection rejected"
        end
        if msg.type == "DTP" and msg.id="accept" then
            return soc
        end
    end
    return nil, "Timeout"
end

ip._handleModemMessage = function(port, headers, ...)
    local soc = ip._sockets[port]
    if soc and type(headers) == "table" and headers.type == "IP" then
        local packet = {...}
        packet.headers = headers
        table.insert(soc._queue, packet)
    end
end

return ip
