--[[FMPM package metadata:
info = "IP configuration utility"
location = "/usr/bin/ipconfig.lua"
]]

local shell = require "shell"
local serialization = require "serialization"
local component = require "component"
local computer = require "computer"
local event = require "event"
local ip = require "ip"

local args, options = shell.parse(...)

if options.h or options.help then
    print("Usage:")
    print("\tipconfig\tDisplay current configuration")
    print("\tipconfig -h")
    print("\t         --help\tDisplays this text")
    print("\tipconfig hostname <hostname>\tSet hostname to <hostname>")
    print("\tipconfig gateway\tInteractive gateway setup")
    print("\tipconfig gateway <gateway>\tSet gateway to <gateway>")
    return
end

local config = ip.getConfig()

function updateConfig(config)
    local file, err = io.open("/etc/ip.conf", "w")
    if file == nil then
        error(err)
    end
    file:write(serialization.serialize(config))
    file:close()
end

if args[1] == "hostname" then
    if args[2] == nil then
        print("Missing hostname")
        return
    end
    if #args[2] > 32 then
        print("Hostname is too long")
        return
    end
    config.hostname = args[2]
    updateConfig(config)
elseif args[1] == "gateway" and args[2] ~= nil then
    config.gateway = args[2]
    updateConfig(config)
elseif args[1] == "gateway" then
    component.modem.broadcast(65535, serialization.serialize{type="discover"})
    local timeout = computer.uptime() + 1
    local candidates = {}
    while computer.uptime() < timeout do
        local name, localAddress, remoteAddress, port, dist, headers, payload = event.pull(timeout - computer.uptime(), "modem_message")
        if port == 65535 and type(headers) == "string" and type(payload) == "string" then
            headers = serialization.unserialize(headers)
            payload = serialization.unserialize(payload)
            if type(headers) == "table" and type(payload) == "table" and headers.type == "routes" then
                for host, hops in pairs(payload) do
                    if hops == 0 then
                        table.insert(candidates, {host = host, addr = remoteAddress})
                    end
                end
            end
        end
    end
    print("Candidates: ")
    for id, c in pairs(candidates) do
        print("\t" .. id .. ") " .. c.host .. "\t" .. c.addr )
    end
    io.write("Choose(1-" .. #candidates .."): ")
    local id
    while type(id) ~= "number" or id < 1 or id > #candidates do
        id = tonumber(io.read())
    end
    config.gateway = candidates[id].addr
    updateConfig(config)
else
    print("Current config:")
    print("\tHostname: " .. (config and config.hostname or "*not set*"))
    print("\tGateway: " .. (config and config.gateway or "*not set*"))
end
