--[[FMPM package metadata:
info = "Ping utility."
location = "/usr/bin/ping.lua"
]]

local component = require "component"
local serialization = require "serialization"
local event = require "event"
local shell = require "shell"
local ip = require "ip"
local computer = require "computer"

local args, options = shell.parse(...)
local rawArgs = {...}

if options.h or options.help or args[1] == nil then
    print("Usage:")
    print("\tping [-t ttl] [-c count] [-s size] [-i interval] <hostname>")
    return
end

local config = ip.getConfig()
local data = ""
local ttl = 32
local count = math.huge
local size = 64
local interval = 1

for k, v in pairs(rawArgs) do
    if v == "-t" then
        ttl = tonumber(rawArgs[k + 1])
    elseif v == "-c" then
        count = tonumber(rawArgs[k + 1])
    elseif v == "-s" then
        size = tonumber(rawArgs[k + 1])
    elseif v == "-i" then
        interval = tonumber(rawArgs[k + 1])
    end
end

for _ = 1, size do
    data = data .. string.char(math.random(0, 255))
end

if not component.modem.isOpen(65535) then
    error("IP daemon is down!")
    return
end

print("PING " .. args[1] .. " " .. #data .. " bytes of data.")

local got = 0

local sent = {}

local id = 0

for _ = 1, count do
    id = id + 1
    component.modem.send(config.gateway, 65535,
        serialization.serialize{type = "IP", ttl = ttl,
            srcPort = 65535, destPort = 65535,
            destHost = args[1], srcHost = config.hostname},
        serialization.serialize{type = "ICMP", id = "ping"},
        id,
        computer.uptime(),
        data)

    sent[id] = computer.uptime()

    local timeout = computer.uptime() + interval
    while computer.uptime() < timeout or id == count do
        local name, localAddress, remoteAddress, port, dist, headers, subheaders, seq, time, payload = event.pull(id == count and timeout - computer.uptime() or math.huge)
        if name == "interrupted" then
            print("^C")
            goto summary
        end
        if name == "modem_message" and port == 65535 and type(headers) == "string" and type(subheaders) == "string" then
            headers = serialization.unserialize(headers)
            subheaders = serialization.unserialize(subheaders)
            if type(headers) == "table" and type(subheaders) == "table" and headers.type == "IP" and subheaders.type == "ICMP" then
                if subheaders.id == "pong" then
                    print(#payload .. " bytes from " .. headers.srcHost ..
                        ": seq=" .. seq .. " ttl=" .. headers.ttl ..
                        " time=" .. math.floor((computer.uptime() - sent[seq]) * 100) .. "ms")
                    got = got + 1
                    if seq == count then
                        goto summary
                    end
                elseif subheaders.id == "HostUnreachable" then
                    print("Host Unreachable")
                elseif subheaders.id == "TtlExpired" then
                    print("TTL Expired (too many hops)")
                end
            end
        end
    end
end

::summary::
print("--- " .. args[1] .. " ping statistics ---")
print(id .. " packets transmitted, " .. got .. " received, " .. math.floor(100 - got/id*100) .. "% packet loss")
